/* ***************************************************************** 
    MESQUITE -- The Mesh Quality Improvement Toolkit

    Copyright 2004 Sandia Corporation and Argonne National
    Laboratory.  Under the terms of Contract DE-AC04-94AL85000 
    with Sandia Corporation, the U.S. Government retains certain 
    rights in this software.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License 
    (lgpl.txt) along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
    diachin2@llnl.gov, djmelan@sandia.gov, mbrewer@sandia.gov, 
    pknupp@sandia.gov, tleurent@mcs.anl.gov, tmunson@mcs.anl.gov      
   
  ***************************************************************** */
/*!
  \file   PS.hpp
  \brief  

  The PS Class implements the steepest descent algorythm in
  order to move a free vertex to an optimal position given an
  ObjectiveFunction object and a QaulityMetric object.

  \author Shankar Sastry
  \date   2011-05-13
*/

#ifndef Mesquite_PS_hpp 
#define Mesquite_PS_hpp

#include "Mesquite.hpp"
#include "VertexMover.hpp"
#include "PatchSetUser.hpp"

#include "MsqFreeVertexIndexIterator.hpp"
#include "MsqTimer.hpp"
#include "MsqDebug.hpp"

#ifdef MSQ_USE_OLD_STD_HEADERS
#  include <memory.h>
#else
#  include <memory>
#endif

using namespace std;

namespace MESQUITE_NS
{
  class ObjectiveFunction;

  /*! \class PS

      This is a very basic implementation of the steepest descent optimization algorythm.
      It works on patches of any size but the step size is hard-wired.
      Obvisouly, this is for testing purposed only. */ 
  class PS : public VertexMover, public PatchSetUser
  {
  public:
    PS(ObjectiveFunction* of, bool Nash = true);

    virtual ~PS() { }
    
    virtual msq_std::string get_name() const;
    
    virtual PatchSet* get_patch_set();
    
    bool project_gradient() const 
      { return projectGradient; }
    
    void project_gradient( bool yesno ) 
      { projectGradient = yesno; }
    
    //bool cosine_projection_step() const 
    //  { return cosineStep; }
    //
    //void cosine_projection_step( bool yesno ) 
    //  { cosineStep = yesno; }
    
  protected:
    virtual void initialize(PatchData &pd, MsqError &err);
    virtual void optimize_vertex_positions(PatchData &pd,
                                         MsqError &err);
    virtual void initialize_mesh_iteration(PatchData &pd, MsqError &err);
    virtual void terminate_mesh_iteration(PatchData &pd, MsqError &err);
    virtual void cleanup();
  
  private:
    bool projectGradient;
    //bool cosineStep;
  };
  
}

#endif















/********************************************************************************************/
namespace MESQUITE_NS {

msq_std::string PS::get_name() const
  { return "PS"; }
  
PatchSet* PS::get_patch_set()
  { return PatchSetUser::get_patch_set(); }

PS::PS(ObjectiveFunction* of, bool Nash) 
  : VertexMover(of, Nash),
    PatchSetUser(true),
    projectGradient(false) //,
    //cosineStep(false)
{
}  
  

void PS::initialize(PatchData &/*pd*/, MsqError &/*err*/)
{
}

void PS::initialize_mesh_iteration(PatchData &/*pd*/, MsqError &/*err*/)
{
}

void PS::optimize_vertex_positions(PatchData &pd, 
                                                MsqError &err)
{
  MSQ_FUNCTION_TIMER( "PS::optimize_vertex_positions" );

  const int SEARCH_MAX = 10;
  const double c1 = 1e-4;
  //msq_std::vector<Vector3D> unprojected(pd.num_free_vertices()); 
  msq_std::vector<Vector3D> gradient(pd.num_free_vertices()); 
  bool feasible=true;//bool for OF values
  double min_edge_len, max_edge_len;
  double step_size, original_value, new_value;
  double norm_squared;
  PatchDataVerticesMemento* pd_previous_coords;
  TerminationCriterion* term_crit=get_inner_termination_criterion();
  OFEvaluator& obj_func = get_objective_function_evaluator();
  
    // get vertex memento so we can restore vertex coordinates for bad steps.
  pd_previous_coords = pd.create_vertices_memento( err ); MSQ_ERRRTN(err);
    // use auto_ptr to automatically delete memento when we exit this function
  msq_std::auto_ptr<PatchDataVerticesMemento> memento_deleter( pd_previous_coords );

    // Evaluate objective function.
    //
    // Always use 'update' version when beginning optimization so that
    // if doing block coordinate descent the OF code knows the set of
    // vertices we are modifying during the optimziation (the subset
    // of the mesh contained in the current patch.)  This has to be
    // done up-front because typically an OF will just store the portion
    // of the OF value (e.g. the numeric contribution to the sum for an
    // averaging OF) for the initial patch.
  feasible = obj_func.update( pd, original_value, gradient, err ); MSQ_ERRRTN(err);
    // calculate gradient dotted with itself
  norm_squared = length_squared( &gradient[0], gradient.size() );
  
    //set an error if initial patch is invalid.
//printf("here\n");
  if(!feasible){
    MSQ_SETERR(err)("PS passed invalid initial patch.",
                    MsqError::INVALID_ARG);
    return;
  }
  

    // use edge length as an initial guess for for step size
  pd.get_minmax_edge_length( min_edge_len, max_edge_len );
  //step_size = max_edge_len / msq_stdc::sqrt(norm_squared);
  //if (!finite(step_size))  // zero-length gradient
  //  return;
  if (norm_squared < DBL_EPSILON)
    return;
  step_size = max_edge_len / msq_std::sqrt(norm_squared) * pd.num_free_vertices();

    // The steepest descent loop...
    // We loop until the user-specified termination criteria are met.
  //cout<<"here\n";
  while (!term_crit->terminate()) {
    MSQ_DBGOUT(3) << "Iteration " << term_crit->get_iteration_count() << msq_stdio::endl;
    MSQ_DBGOUT(3) << "  o  original_value: " << original_value << msq_stdio::endl;
    MSQ_DBGOUT(3) << "  o  grad norm suqared: " << norm_squared << msq_stdio::endl;

      // Save current vertex coords so that they can be restored if
      // the step was bad.
    pd.recreate_vertices_memento( pd_previous_coords, err ); MSQ_ERRRTN(err);
	
	// Reduce step size until it satisfies Armijo condition
    //cout<<"here\n";
    int counter = 0, max_index=-1;
    double mystep_size[6];
    for(int j=0; j<6; j++)
	mystep_size[j] = step_size;

	for (;;) {
		if (++counter > SEARCH_MAX || step_size < DBL_EPSILON) {
			//cout<<"Worked hard!\n";
			MSQ_DBGOUT(3) << "    o  No valid step found.  Giving Up." << msq_stdio::endl;
			return;
		}
			
		double curr_max;
		max_index=-1;
		curr_max = original_value;
		for(int dir = 0; dir<6; dir++){
			//cout<<"dir = "<<dir<<", step_size = "<<mystep_size[dir]<<", counter = "<<counter<<endl;
			switch (dir){
				case 0:gradient[0].set(0,0,1); break;
				case 1:gradient[0].set(0,0,-1); break;
				case 2:gradient[0].set(0,1,0); break;
				case 3:gradient[0].set(0,-1,0); break;
				case 4:gradient[0].set(1,0,0); break;
				case 5:gradient[0].set(-1,0,0); break;
			}

/*****************************NEW************************************************/
/*
vector<Vector3D> vertex1;
Vector3D vertex;
pd.get_free_vertex_coordinates(vertex1);
vertex = vertex1[0];
Vector3D grad;
  Vector3D points[20][3];
  points[0][0].set(1,0,1);
  points[0][1].set(1,0,0);
  points[0][2].set(0.404509,0.293893,1.000000);

  points[10][0].set(0.404509,0.293893,0.000000);
  points[10][1].set(1,0,0);
  points[10][2].set(0.404509,0.293893,1.000000);

  points[1][0].set(0.309017, 0.951057, 1.000000);
  points[1][1].set(0.309017, 0.951057, 0.000000);
  points[1][2].set(0.404509, 0.293893, 1.000000);

  points[11][0].set(0.404509, 0.293893, 0.000000);
  points[11][1].set(0.309017, 0.951057, 0.000000);
  points[11][2].set(0.404509, 0.293893, 1.000000);
  
  points[2][0].set(-0.809017, 0.587785, 1.000000);
  points[2][1].set(-0.809017, 0.587785, 0.000000);
  points[2][2].set(-0.154509, 0.475528, 1.000000);

  points[12][0].set(-0.154509, 0.475528, 0.000000);
  points[12][1].set(-0.809017, 0.587785, 0.000000);
  points[12][2].set(-0.154509, 0.475528, 1.000000);

  points[3][0].set(-0.809017, 0.587785, 1.000000);
  points[3][1].set(-0.809017, 0.587785, 0.000000);
  points[3][2].set(-0.500000, 0.000000, 1.000000);

  points[13][0].set(-0.500000, 0.000000, 0.000000);
  points[13][1].set(-0.809017, 0.587785, 0.000000);
  points[13][2].set(-0.500000, 0.000000, 1.000000);

  points[4][0].set(0.309017, 0.951057, 1.000000);
  points[4][1].set(0.309017, 0.951057, 0.000000);
  points[4][2].set(-0.154509, 0.475528, 1.000000);

  points[14][0].set(-0.154509, 0.475528, 0.000000);
  points[14][1].set(0.309017, 0.951057, 0.000000);
  points[14][2].set(-0.154509, 0.475528, 1.000000);

  points[5][0].set(-0.809017, -0.587785, 1.000000);
  points[5][1].set(-0.809017, -0.587785, 0.000000);
  points[5][2].set(-0.500000, 0.000000, 1.000000);

  points[15][0].set(-0.500000, 0.000000, 0.000000);
  points[15][1].set(-0.809017, -0.587785, 0.000000);
  points[15][2].set(-0.500000, 0.000000, 1.000000);

  points[6][0].set(-0.809017, -0.587785, 1.000000);
  points[6][1].set(-0.809017, -0.587785, 0.000000);
  points[6][2].set(-0.154509, -0.475528, 1.000000);

  points[16][0].set(-0.154509, -0.475528, 0.000000);
  points[16][1].set(-0.809017, -0.587785, 0.000000);
  points[16][2].set(-0.154509, -0.475528, 1.000000);

  points[7][0].set(-0.154509, -0.475528, 1.000000);
  points[7][1].set(-0.154509, -0.475528, 0.000000);
  points[7][2].set(0.309017, -0.951057, 1.000000);

  points[17][0].set(0.309017, -0.951057, 0.000000);
  points[17][1].set(-0.154509, -0.475528, 0.000000);
  points[17][2].set(0.309017, -0.951057, 1.000000);

  points[8][0].set(0.309017, -0.951057, 1.000000);
  points[8][1].set(0.309017, -0.951057, 0.000000);
  points[8][2].set(0.404509, -0.293893, 1.000000);

  points[18][0].set(0.404509, -0.293893, 0.000000);
  points[18][1].set(0.309017, -0.951057, 0.000000);
  points[18][2].set(0.404509, -0.293893, 1.000000);

  points[9][0].set(0.404509, -0.293893, 1.000000);
  points[9][1].set(0.404509,-0.293893,0.000000);
  points[9][2].set(1,0,1);

  points[19][0].set(1,0,0);
  points[19][1].set(0.404509,-0.293893,0.000000);
  points[19][2].set(1,0,1);
  
  
  for(int numfree = 0; numfree<1; numfree++)
  {
	int count=0;
	bool pl = 0;
	for(int j =0; j<3; j++)
		grad = gradient[0];
	Vector3D origgrad = grad;
	if(vertex[2] == 0 || vertex[2] == 1)
  	{
		//count++;
		pl = 1;
  		grad[2]=0;
		//printf("here we are\n");
  	}
	/*for(int i=0; i<20; i++)
	{
		//double vol=(points[i][1]-points[i][0])%((points[i][2]-points[i][0])*(vertex-points[i][0]));
		double vol=(vertex-points[i][0])%((points[i][2]-points[i][0])*(points[i][1]-points[i][0]));
		if (fabs(vol) < 1e-2)
		{
			if(i<10)
			{
				if(points[i][0][0]<points[i][2][0])
				{
					if(vertex[0]<points[i][0][0] ||vertex[0]>points[i][2][0])
						continue;
				}
				else
				{
					if(vertex[0]>points[i][0][0] ||vertex[0]<points[i][2][0])
						continue;
				}
			}
			else
			{
				if(points[i][0][0]<points[i][1][0])
				{
					if(vertex[0]<points[i][0][0] ||vertex[0]>points[i][1][0])
						continue;
				}
				else
				{
					if(vertex[0]>points[i][0][0] ||vertex[0]<points[i][1][0])
						continue;
				}
			}
			Vector3D normal;
			normal = (points[i][1]-points[i][0])*(points[i][2]-points[i][0]);
			//double orig  = grad%normal;
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			//printf("%lf %lf %lf\n", points[i][0][0], points[i][0][1], points[i][0][2]);
			//printf("%lf %lf %lf\n", points[i][1][0], points[i][1][1], points[i][1][2]);
			//printf("%lf %lf %lf\n", points[i][2][0], points[i][2][1], points[i][2][2]);
			//printf("%.10lf %.10lf %10lf\n", orig, grad%normal, orig/(grad%normal));
			if ((grad%normal) > 1e-7)
				printf("problem\n");	
			count++;
		} 
	}
	for(int j=0; j<3; j++)
		gradient[0] = grad;
	if(count>2)
	{
		gradient[0][0]=0;
		gradient[0][1]=0;
	}
	//if(count>4 && pl)
		//printf("%d %lf %lf %lf %lf %lf %lf\n", count, origgrad[0], origgrad[1],origgrad[2], grad[0], grad[1], grad[2]);
  }

 */ 
/*****************************NEW************************************************/
			//cout<<feasible<<endl;
			feasible = false;
			int count=0;
			while(!feasible && count<10)
			{
				count++;
				pd.move_free_vertices_constrained( &gradient[0], gradient.size(), -mystep_size[dir], err ); MSQ_ERRRTN(err);
				// Evaluate objective function for new vertices.  We call the
				// 'evaluate' form here because we aren't sure yet if we want to
				// keep these vertices.  Until we call 'update', we have the option
				// of reverting a block coordinate decent objective function's state
				// to that of the initial vertex coordinates.  However, for block
				// coordinate decent to work correctly, we will need to call an
				// 'update' form if we decide to keep the new vertex coordinates.
				feasible = obj_func.evaluate( pd, new_value, err ); MSQ_ERRRTN(err);
				MSQ_DBGOUT(3) << "    o  step_size: " << step_size << msq_stdio::endl;
				MSQ_DBGOUT(3) << "    o  new_value: " << new_value << msq_stdio::endl;
				//cout<<feasible<<" "<<new_value<<endl;
				if (!feasible) {
					// OF value is invalid, decrease step_size a lot
					mystep_size[dir] *= 0.75;
					//cout<<mystep_size[dir]<<endl;
					pd.set_to_vertices_memento( pd_previous_coords, err );  MSQ_ERRRTN(err);
				}
				else if (new_value < curr_max) {
					max_index=dir;
					curr_max = new_value;
				}
			}
						
			// undo previous step : restore vertex coordinates
			pd.set_to_vertices_memento( pd_previous_coords, err );  MSQ_ERRRTN(err);
		}
		if(max_index!=-1)
		{
			break;
		}
		else
		{
			step_size *= 0.5;
			for (int j=0; j<6; j++)
				mystep_size[j]*=0.5;
		}
	}
	if(max_index!=-1)
	{
		switch (max_index){
			case 0:gradient[0].set(0,0,1); break;
			case 1:gradient[0].set(0,0,-1); break;
			case 2:gradient[0].set(0,1,0); break;
			case 3:gradient[0].set(0,-1,0); break;
			case 4:gradient[0].set(1,0,0); break;
			case 5:gradient[0].set(-1,0,0); break;
		}
/*****************************NEW************************************************/
/*
vector<Vector3D> vertex1;
Vector3D vertex;
pd.get_free_vertex_coordinates(vertex1);
vertex = vertex1[0];
Vector3D grad;
  Vector3D points[20][3];
  points[0][0].set(1,0,1);
  points[0][1].set(1,0,0);
  points[0][2].set(0.404509,0.293893,1.000000);

  points[10][0].set(0.404509,0.293893,0.000000);
  points[10][1].set(1,0,0);
  points[10][2].set(0.404509,0.293893,1.000000);

  points[1][0].set(0.309017, 0.951057, 1.000000);
  points[1][1].set(0.309017, 0.951057, 0.000000);
  points[1][2].set(0.404509, 0.293893, 1.000000);

  points[11][0].set(0.404509, 0.293893, 0.000000);
  points[11][1].set(0.309017, 0.951057, 0.000000);
  points[11][2].set(0.404509, 0.293893, 1.000000);
  
  points[2][0].set(-0.809017, 0.587785, 1.000000);
  points[2][1].set(-0.809017, 0.587785, 0.000000);
  points[2][2].set(-0.154509, 0.475528, 1.000000);

  points[12][0].set(-0.154509, 0.475528, 0.000000);
  points[12][1].set(-0.809017, 0.587785, 0.000000);
  points[12][2].set(-0.154509, 0.475528, 1.000000);

  points[3][0].set(-0.809017, 0.587785, 1.000000);
  points[3][1].set(-0.809017, 0.587785, 0.000000);
  points[3][2].set(-0.500000, 0.000000, 1.000000);

  points[13][0].set(-0.500000, 0.000000, 0.000000);
  points[13][1].set(-0.809017, 0.587785, 0.000000);
  points[13][2].set(-0.500000, 0.000000, 1.000000);

  points[4][0].set(0.309017, 0.951057, 1.000000);
  points[4][1].set(0.309017, 0.951057, 0.000000);
  points[4][2].set(-0.154509, 0.475528, 1.000000);

  points[14][0].set(-0.154509, 0.475528, 0.000000);
  points[14][1].set(0.309017, 0.951057, 0.000000);
  points[14][2].set(-0.154509, 0.475528, 1.000000);

  points[5][0].set(-0.809017, -0.587785, 1.000000);
  points[5][1].set(-0.809017, -0.587785, 0.000000);
  points[5][2].set(-0.500000, 0.000000, 1.000000);

  points[15][0].set(-0.500000, 0.000000, 0.000000);
  points[15][1].set(-0.809017, -0.587785, 0.000000);
  points[15][2].set(-0.500000, 0.000000, 1.000000);

  points[6][0].set(-0.809017, -0.587785, 1.000000);
  points[6][1].set(-0.809017, -0.587785, 0.000000);
  points[6][2].set(-0.154509, -0.475528, 1.000000);

  points[16][0].set(-0.154509, -0.475528, 0.000000);
  points[16][1].set(-0.809017, -0.587785, 0.000000);
  points[16][2].set(-0.154509, -0.475528, 1.000000);

  points[7][0].set(-0.154509, -0.475528, 1.000000);
  points[7][1].set(-0.154509, -0.475528, 0.000000);
  points[7][2].set(0.309017, -0.951057, 1.000000);

  points[17][0].set(0.309017, -0.951057, 0.000000);
  points[17][1].set(-0.154509, -0.475528, 0.000000);
  points[17][2].set(0.309017, -0.951057, 1.000000);

  points[8][0].set(0.309017, -0.951057, 1.000000);
  points[8][1].set(0.309017, -0.951057, 0.000000);
  points[8][2].set(0.404509, -0.293893, 1.000000);

  points[18][0].set(0.404509, -0.293893, 0.000000);
  points[18][1].set(0.309017, -0.951057, 0.000000);
  points[18][2].set(0.404509, -0.293893, 1.000000);

  points[9][0].set(0.404509, -0.293893, 1.000000);
  points[9][1].set(0.404509,-0.293893,0.000000);
  points[9][2].set(1,0,1);

  points[19][0].set(1,0,0);
  points[19][1].set(0.404509,-0.293893,0.000000);
  points[19][2].set(1,0,1);
  
  
  for(int numfree = 0; numfree<1; numfree++)
  {
	int count=0;
	bool pl = 0;
	for(int j =0; j<3; j++)
		grad = gradient[0];
	Vector3D origgrad = grad;
	if(vertex[2] == 0 || vertex[2] == 1)
  	{
		//count++;
		pl = 1;
  		grad[2]=0;
		//printf("here we are\n");
  	}
	for(int i=0; i<20; i++)
	{
		//double vol=(points[i][1]-points[i][0])%((points[i][2]-points[i][0])*(vertex-points[i][0]));
		double vol=(vertex-points[i][0])%((points[i][2]-points[i][0])*(points[i][1]-points[i][0]));
		if (fabs(vol) < 1e-2)
		{
			if(i<10)
			{
				if(points[i][0][0]<points[i][2][0])
				{
					if(vertex[0]+1e-6<points[i][0][0] ||vertex[0]-1e-6>points[i][2][0])
						continue;
				}
				else
				{
					if(vertex[0]-1e-6>points[i][0][0] ||vertex[0]+1e-6<points[i][2][0])
						continue;
				}
			}
			else
			{
				if(points[i][0][0]<points[i][1][0])
				{
					if(vertex[0]+1e-6<points[i][0][0] ||vertex[0]-1e-6>points[i][1][0])
						continue;
				}
				else
				{
					if(vertex[0]-1e-6>points[i][0][0] ||vertex[0]+1e-6<points[i][1][0])
						continue;
				}
			}
			Vector3D normal;
			normal = (points[i][1]-points[i][0])*(points[i][2]-points[i][0]);
			//double orig  = grad%normal;
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
				+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			//printf("%lf %lf %lf\n", points[i][0][0], points[i][0][1], points[i][0][2]);
			//printf("%lf %lf %lf\n", points[i][1][0], points[i][1][1], points[i][1][2]);
			//printf("%lf %lf %lf\n", points[i][2][0], points[i][2][1], points[i][2][2]);
			//printf("%.10lf %.10lf %10lf\n", orig, grad%normal, orig/(grad%normal));
			if ((grad%normal) > 1e-7)
				printf("problem\n");	
			count++;
		} 
	}
	for(int j=0; j<3; j++)
		gradient[0] = grad;
	if(count>2)
	{
		gradient[0][0]=0;
		gradient[0][1]=0;
	}
	//if(count>4 && pl)
		//printf("%d %lf %lf %lf %lf %lf %lf\n", count, origgrad[0], origgrad[1],origgrad[2], grad[0], grad[1], grad[2]);
  }
*/
  
/*****************************NEW************************************************/
		pd.move_free_vertices_constrained( &gradient[0], gradient.size(), -mystep_size[max_index], err ); MSQ_ERRRTN(err);
	}
	else
	{	
		break;
	}
      // Re-evaluate objective function to get gradient.
      // Calling the 'update' form here incorporates the new vertex 
      // positions into the 'accumulated' value if we are doing a 
      // block coordinate descent optimization.
    //obj_func.update(pd, original_value, gradient, err ); MSQ_ERRRTN(err);
	obj_func.update(pd, original_value, err ); MSQ_ERRRTN(err);
    
      // Update terination criterion for next iteration.
      // This is necessary for efficiency.  Some values can be adjusted
      // for each iteration so we don't need to re-caculate the value
      // over the entire mesh.
    term_crit->accumulate_patch( pd, err );  MSQ_ERRRTN(err);
    term_crit->accumulate_inner( pd, original_value, &gradient[0], err ); MSQ_ERRRTN(err); 
      
      // Calculate initial step size for next iteration using step size 
      // from this iteration
    /*step_size *= norm_squared;
    norm_squared = length_squared( &gradient[0], gradient.size() );
    if (norm_squared < DBL_EPSILON)
      break;
    step_size /= norm_squared;*/
    step_size *= 20;
  }
}

void PS::terminate_mesh_iteration(PatchData &/*pd*/, MsqError &/*err*/)
{
  //  cout << "- Executing PS::iteration_complete()\n";
}
  
void PS::cleanup()
{
  //  cout << "- Executing PS::iteration_end()\n";
}
  
} // namespace Mesquite

